#ifndef ER_EXACT_LS_H
#define ER_EXACT_LS_H

#include <fmpq.h>
#include <fmpq_mat.h>
#include <fmpz.h>
#include <fmpz_mat.h>

int factoring(int m, int n, double A[], fmpz_mat_t U, fmpz_mat_t L, fmpz_mat_t D, int Pr[], int Pc[]);
int factoring(int m, int n, fmpz_mat_t A, fmpz_mat_t U, fmpz_mat_t L, fmpz_mat_t D, int Pr[], int Pc[]);
/** *****************************************************************
    Carry out Gaussian LU decomposition with exact arithmetic using
    the Fraction-free method on the matrix {A}, where {A} is a known
    matrix of size {m x n}. Matrix {A} is decomposed into five
    matrices, so that { A = Pr * L * D^(-1) * U * Pc}, where {Pr}
    is a permutation matrix of rows, {L} is a lower triangular matrix,
    {D} is a diagonal matrix, {U} is a upper triangular matrix and
    {Pc} is a permutation matrix of columns.
    The matrix {A} is not modified.

    Returns the number {r} of linearly independent rows of the
    matrix {A}, and {-1} if it was not possible obtain a
    triangularization of the matrix {A}.
*********************************************************************/


void defineB(int r, int m, int p, int Pr[], fmpz_mat_t QE, fmpz_mat_t REA, fmpz_mat_t PhiA, fmpz_mat_t REF, fmpz_mat_t PhiF, fmpz_mat_t b, fmpz_mat_t B, fmpz_mat_t B_pr);
/** ******************************************************************

    Return vetor B.
**********************************************************************/


bool verifySolvability(int r, int m, fmpz_mat_t L, int p, fmpz_mat_t B);
/** ******************************************************************
    Verify if the system in question has solution by the veracity of
    the equation {L2 * L1^(-1) * B1 = B2}, where {L1}, of the size {r x r},
    and {L2}, of the size {m-r x r} are parts of a known lower
    triangular matrix {L}, and {B1}, of the size {r x p}, and {B2},
    of the size {m-r x p} are parts of a known matrix {B}.

    Return {true} or {false}.
**********************************************************************/


void permuteRows(int m, int n, int Pr[], fmpz_mat_t A, fmpz_mat_t A_pr);
/** ******************************************************************
    Execute the permutation of the rows of the matrix {A}
    multiplying a known permutation matrix of the rows {Pr} by
    the matrix {A}. The matrix {Pr} is not modified.

    Return the permuted matrix {A}.
*********************************************************************/


void permuteColumns(int m, int n, int Pc[], fmpz_mat_t A, fmpz_mat_t A_pc);
/** ******************************************************************
    Execute the permutation of the columns of the matrix {A}
    multiplying a known permutation matrix of the columns {Pc} by
    the matrix {A}. The matrix {Pc} is not modified.

    Return the permuted matrix {A}.
*********************************************************************/


void computeInverseVector(int n, int P[], int Q[]);
/** ********************************************************************
  Calculate the inverse vector of the vector {P}.

  Return the inverse vector {Q}.
***********************************************************************/

// ------------------------------------------------------------------------------------------------------------------ //
// -------------------------------------- FUNCTIONS NOT USED BY ECLES ----------------------------------------------- //
// ------------------------------------------------------------------------------------------------------------------ //


int solveSystem(int n, fmpz_mat_t A, int a, fmpz_mat_t B, double Xd[]);
int solveSystem(int n, fmpz_mat_t A, int a, fmpz_mat_t B, fmpz_mat_t X, fmpz_t den);
/** *******************************************************************
    Solves the linear system {A X = B}, where {A} is a known matrix of
    size {m x n}, {B} is a known matrix of size {m x a}, and {X} is an
    unknown matrix of size {n x a}. The matrices {A} an d {B} are not
    modified.

    Return the matrix {X}, the exact solution of the linear system.
***********************************************************************/


int solveY(int m, int r, fmpz_mat_t D, fmpz_mat_t L, int p, fmpz_mat_t B1, fmpz_mat_t Y);
/** *******************************************************************
    Solves the equation {Y = D * L^(-1) * B1}, where {D} is a known
    diagonal matrix of size {r x r}, {L-1} is the inverse matrix
    of a known lower triangular matrix {L} of size {r x r}, and {B1}
    is a known matrix of size {m x a}, and {Y} is an unknown matrix
    of size {r x p}. The matrices {D}, {L} and {B1} are not modified.

    Return the matrix {Y}, the exact solution of the equation
    {Y = D * L^(-1) * B1}.
***********************************************************************/


int solveX(int r, int n, fmpz_mat_t U, int p, fmpz_mat_t Y, double Xd[]);
int solveX(int r, int n, fmpz_mat_t U, int p, fmpz_mat_t Y, fmpq_mat_t X);
/** ********************************************************************
    Solves the equation {X = U1^(-1) * Y}, where {U1^(-1)} is the inverse
    matrix {U1} of the size {r x r}, obtained separating a known upper
    triangular matrix {U} of size {r x n} in two matrices {U1} and {U2},
    and {Y} is the solution matrix of size {r x a} of the equation
    {Y = D * L^(-1) * B1}, and {X} is an unknown matrix of size {r x a}.
    The matrices {U} and {Y} are not modified.

    Obs: the columns of the matrix {U2} can take any value, because
    this parameters are free.

    Return the matrix {X}, the exact solution of the equation
    {X = U1^(-1) * Y}.
***********************************************************************/


// ------------------------------------------------------------------------------------------------------------------ //
// ------------------------------------------ AUXILIARY FUNCTIONS --------------------------------------------------- //
// ------------------------------------------------------------------------------------------------------------------ //


void initializeMatrices(int aux, fmpz_mat_t AB, fmpz_mat_t U_aux, fmpz_mat_t L_aux, fmpz_mat_t D_aux, fmpz_mat_t V);
/** ********************************************************************
  Initialize matrices used in the {rank()} function.
************************************************************************/

void initPermutationVector(int m, int *P);
/** ********************************************************************
  Initialize a permutation vector.
************************************************************************/

int splitRows(int m, int r, int n, fmpz_mat_t A, fmpz_mat_t A1);
int splitRows(int m, int r, int n, fmpz_mat_t A, fmpz_mat_t A1, fmpz_mat_t A2);
/** ******************************************************************
    Copy the {r} first rows of the matrix {A} to matrix {A1}, and
    copy the last {m-r} rows of the matrix {A} to matrix {A2}.
    The matrix {A} is not modified.

    Return the matrix {A1} linearly independent and the matrix {A2}
    with the redundant rows of the matrix {A}.
*********************************************************************/


int splitColumns(int r, int n, fmpz_mat_t A, fmpz_mat_t A1);
int splitColumns(int r, int n, fmpz_mat_t A, fmpz_mat_t A1, fmpz_mat_t A2);
/** ******************************************************************
    Copy the {r} first columns of the matrix {A} to matrix {A1}, and
    copy the last {m-r} columns of the matrix {A} to matrix {A2}.
    The matrix {A} is not modified.

    Return the matrix {A1} linearly independent and the matrix {A2}
    with the redundant columns of the matrix {A}.
*********************************************************************/


void convertFmpqToDoubleMatrix(int m, int n, fmpq_mat_t M, double *Md);

void convertFmpzToDoubleMatrix(int m, int n, fmpz_mat_t M, double *Md, fmpz_t d);
/** *******************************************************************
    Convert a multiprecision integer matrix {M} to a float point
    matrix {Md}, both of the size {m x n} and a denominator {d}.
    The matrix {M} is not modified.

    Return the matrix {Md}, whose the coefficients are float point.
***********************************************************************/

void convertMatrixToExact (int m, int n, double *Md, fmpz_mat_t M);
/** *******************************************************************
    Convert a float point matrix {Md} to a multiprecision integer
    matrix {M}, both of the size {m x n}. The matrix {Md} is not
    modified.

    Return the matrix {M}, whose the coefficients are multiprecision
    integers.
***********************************************************************/


void printPermutationVector(int m, int Pr[]);
/** ********************************************************************
    Print a permutation vector.
************************************************************************/

void gsel_print_array(FILE *wr, char *fmt, char *head, int m, int n, double M[], char *foot);
/** ********************************************************************************************
    Writes to {wr} the array {M}, in a human-readable format. Each
    element is printed with format {fmt}. The strings {head} and
    {foot}, if not NULL, are printed on separate lines before and
    after the array.
************************************************************************************************/

double *rmxn_alloc(int m, int n);
/** ********************************************************************************************
    Allocates {m*n} {double}s on the heap; bombs out if no mem.
************************************************************************************************/



#endif // ER_EXACT_LS_H
