/********************************************************************
*                         ELO MALUCO                                *
* ----------------------------------------------------------------- *
*      Desenvolvedores:                                             *
*               Anderson de Rezende Rocha - DCC/UFLA                *
*               undersun@comp.ufla.br                               *
*                                                                   *
*               Jlio Csar Alves - DCC/UFLA                        *
*               jcalves@comp.ufla.br                                *
* ----------------------------------------------------------------- *
*    COMPUTACAO GRAFICA: Prof. Bruno Oliveira Schneider             *
********************************************************************/

/********************************************************************
*  Este arquivo contem a classe clEloMaluco que define o Elo Maluco *
*  em si.                                                           *
*  Ela utiliza o enum E_MOVIMENTO, que representa os movimentos     *
*  ao jogar.                                                        *
********************************************************************/

#ifndef ELO_MALUCO_H
#define ELO_MAULCO_H

// fstream -> utilizada para trabalhar com arquivos
#include <fstream>
// face.h -> define uma face do elo
#include "face.h"
// lista.h -> utilizada na lista de movimentos para a solucao automatica
#include "lista.h"
// time.h -> utilizada para gerar numeros "aleatorios" para embaralhar o elo
#include "time.h"

using namespace std;
using std::ifstream;
using std::ofstream;

// enum E_MOVIMENTO
// Representa os movimentos possiveis ao jogar
// RSD -> Rotacionar parte superior para a direita
// RSE -> Rotacionar parte superior para a esquerda
// RID -> Rotacionar parte inferior para a direita
// RIE -> Rotacionar parte inferior para a esquerda
enum E_MOVIMENTO { RSD, RSE, RID, RIE, MFC, MFB, };

// Definicao da classe clEloMaluco
class clEloMaluco
{
    public:
        clEloMaluco();
        ~clEloMaluco() {}

        // carrega o estado do elo a partir de um arquivo
        bool CarregarEstado(char *nomeDoArquivo);
        // salva o estado do elo em um arquivo
        void SalvarEstado(char *nomeDoArquivo);

        // Rotaciona a parte superior para a direita
        void RodarSupDir();
        // Rotaciona a parte superior para a esquerda
        void RodarSupEsq();
        // Rotaciona a parte inferior para a direita
        void RodarInfDir();
        // Rotaciona a parte inferior para a esquerda
        void RodarInfEsq();
        // Move a face que esta abaixo da face vazia para cima
        bool MoverFacePCima();
        // Move a face que esta acima da face vazia para baixo
        bool MoverFacePBaixo();

        // devolve a cabeca da lista de movimentos criada quando
        // se gera a solucao automatica
        clNohLista<E_MOVIMENTO> *CabecaListaSolucao() const;
        // diz se o elo esta solucionado
        bool EstaSolucionado();
        // embaralha as faces do elo
        void Embaralhar();
        // gera um arquivo contendo a solucao para o estado atual
        bool GerarSolucao();
        // a partir de um id devolve uma face por referencia
        bool CopiarFace(int id, clFace &face);


    private:
        // funcao auxiliar para carregar o estado do elo, utilizada tanto
        // pela CarregarEstado(...) quanto pela CarregarSolucao(...)
        bool CarregarEstadoAux(ifstream &arquivo);
        // Carrega a representacao do elo a partir de um arquivo. A representacao
        // sao os pontos e os poligonos que definem o elo
        void CarregarRepresentacao(char *nomeDoArquivo);
        // carrega um arquivo contendo um estado inicial do elo e os passos para a solucao
        bool CarregarSolucao(char *nomeDoArquivo);
        // encontra em qual posicao do vetor de faces esta a face vazia
        int EncontrarPosVazio();
        // Funcao auxiliar para mover a face para cima ou para baixo
        bool MoverFaceAux(bool baixo);
        // funcao auxiliar para rotacionar as partes superior e inferior do elo
        // para a esquerda ou para a direita
        void RodarAux(bool sup, bool dir);
        // retorna se uma coluna do elo esta solucionada
        bool ColunaCorreta(int coluna);

        // numero de faces do elo maluco = 16
        int numeroDeFaces;
        // lista que contera os movimentos para se chegar a solucao
        clLista<E_MOVIMENTO> listaSolucao;
        // vetor que contem as 16 faces do elo
        clFace elo[16];
};

clEloMaluco::clEloMaluco()
{
    // inicializa o numero de faces
    numeroDeFaces = 16;
    // carrega a representacao geometrica do elo a partir do arquivo 'elo.dat'
    CarregarEstado("estado.elo");
}

// carrega o estado do elo a partir de um arquivo
bool clEloMaluco::CarregarEstado(char * nomeDoArquivo)
{
    // cria um objeto da classe ifstream para ser passado para a funcao carregarEstadoAux
    ifstream arquivo(nomeDoArquivo);
    if (!arquivo)
    {
        wxString str("No foi possvel abrir arquivo ");
        str << nomeDoArquivo;
        wxMessageBox(str, "Arquivo", wxOK | wxICON_ERROR);
        return false;
    }
    else // se o arquivo foi aberto corretamente
    {
        if (!CarregarEstadoAux(arquivo))
        {
            wxString str("O arquivo ");
            str << nomeDoArquivo;
            str << " no est no formato correto.\nEstado carregado incorretamente.";
            wxMessageBox(str, "Erro na Descrio", wxOK | wxICON_ERROR);
            return false;
        }
    }
    return true;
}

// salva o estado do elo em um arquivo
void clEloMaluco::SalvarEstado(char * nomeDoArquivo)
{
    // cria um arquivo de saida
    ofstream arquivo(nomeDoArquivo);

    if(!arquivo)
    {
        wxString str("No foi possvel criar ou abrir arquivo ");
        str << nomeDoArquivo;
        wxMessageBox(str, "Arquivo", wxOK | wxICON_ERROR);
    }
    else // se foi possivel criar arquivo
    {
        // loop que escreve o arquivo
        for(int i = 0; i < numeroDeFaces; i++)
        {
            switch(elo[i].GetIdTextura())
            {
                case VMS:
                    arquivo << "vms ";
                    break;
                case VMM:
                    arquivo << "vmm ";
                    break;
                case VMI:
                    arquivo << "vmi ";
                    break;
                case VDS:             
                    arquivo << "vds ";
                    break;
                case VDM:
                    arquivo << "vdm ";
                    break;
                case VDI:
                    arquivo << "vdi ";
                    break;
                case AMS:
                    arquivo << "ams ";
                    break;
                case AMM:
                    arquivo << "amm ";
                    break;
                case AMI:             
                    arquivo << "ami ";
                    break;
                case BRS:
                    arquivo << "brs ";
                    break;
                case BRM:
                    arquivo << "brm ";
                    break;
                case BRI:
                    arquivo << "bri ";
                    break;
                case VZO:             
                    arquivo << "vzo ";
                    break;
                default: 
                    arquivo << " Erro na textura";
            }
            // para escrever uma parte em cada linha
            if(i%4 == 3)
                arquivo << "\n";
        }
    }    
}

// Carrega a representacao do elo a partir de um arquivo. A representacao
// sao os pontos e os poligonos que definem o elo
void clEloMaluco::CarregarRepresentacao(char *nomeDoArquivo)
{
    // arquivo que contem a representacao
    ifstream arquivo(nomeDoArquivo);
    // conjunto de pontos que definem o elo maluco
    clPonto ponto[20];
    // coordenadas de um ponto
    double x, y, z;
    // inicializacao das coordenadas com zero
    x = y = z = 0.0;
    // identificadores de pontos
    int pt1, pt2, pt3, pt4;
    pt1 = pt2 = pt3 = pt4 = 0;


    if (!arquivo)
    {
        wxString mensagem("No foi possvel abrir o arquivo ");
        mensagem << nomeDoArquivo;
        wxMessageBox(mensagem, "Erro de leitura", wxOK | wxICON_ERROR);
    }
    else // se o arquivo foi corretamente aberto
    {
        // laco que le os 20 pontos que definem e elo maluco
        // os pontos serao identificados pela ordem que aparecem no arquivo
        // o primeiro eh o ponto 1, o segundo o ponto 2, e assim por diante
        for (int i = 0; i < 20; i++)
        {
            // le as tres coordenadas...
            arquivo >> x >> y >> z;
            // ... e seta o ponto com estes valores
            ponto[i].SetX(x/100);
            ponto[i].SetY(y/100);
            ponto[i].SetZ(z/100);
        }
        // laco que le os os identificadores dos pontos que formam os 16 poligonos
        // um para cada face
        for (int i = 0; i < 16; i++)
        {
            // le quatro identificadores de pontos
            arquivo >> pt1 >> pt2 >> pt3 >> pt4;
            // seta os pontos que formam uma face do elo
            elo[i].SetPonto1(ponto[pt1-1]);
            elo[i].SetPonto2(ponto[pt2-1]);
            elo[i].SetPonto3(ponto[pt3-1]);
            elo[i].SetPonto4(ponto[pt4-1]);
        }
    }
}

// carrega um arquivo contendo um estado inicial do elo e os passos para a solucao
bool clEloMaluco::CarregarSolucao(char *nomeDoArquivo)
{
    // contera cada token lido do arquivo
    char str[4];
    // utilizada para se ignorar as linhas com comentarios (consideramos 100 um tamanho razoavel)
    char lixo[100];

    // arquivo de entrada
    ifstream arquivo(nomeDoArquivo);
    if (!arquivo)
    {
        wxString str("No foi possvel abrir arquivo ");
        str << nomeDoArquivo;
        wxMessageBox(str, "Arquivo", wxOK | wxICON_ERROR);
        return false;
    }
    else
    {
        // carrega o estado inicial presente no inicio do arquivo
        if (!CarregarEstadoAux(arquivo))
        {
            wxString str("O arquivo ");
            str << nomeDoArquivo;
            str << " no est no formato correto.\nEstado carregado incorretamente.";
            wxMessageBox(str, "Erro na Descrio", wxOK | wxICON_ERROR);
            return false;
        }
        else // se o estado inicial foi carregado com sucesso
        {
            // enquanto se consegue ler um token do arquivo, coloca-se os movimentos para a solucao
            // em uma lista
            while (!arquivo.eof() && arquivo >> str)
            {
                // ignora linhas com comentarios
                if(str[0] == '#')
                    arquivo.getline(lixo, sizeof(lixo));
                else
                {
                    // coloca um identificador do movimento na lista
                    if(str == (char *)"rse")
                        listaSolucao.InserirNoFim(RSE);
                    else if(str == (char *)"rsd")
                        listaSolucao.InserirNoFim(RSD);
                    else if(str == (char *)"rie")
                        listaSolucao.InserirNoFim(RIE);
                    else if(str == (char *)"rid")
                        listaSolucao.InserirNoFim(RID);
                    else if(str == (char *)"mfc")
                        listaSolucao.InserirNoFim(MFC);
                    else if(str == (char *)"mfb")
                        listaSolucao.InserirNoFim(MFB);
                    else
                    {
                        // caso o arquivo nao esteja no formato correto
                        listaSolucao.Limpar();                   
                        wxMessageBox("Arquivo de soluo mal gerado.\n"
                                     "Impossvel mostrar passos da soluo.",
                                     "Elo Maluco", wxOK | wxICON_INFORMATION);
                        return false;
                    }
                }
            }               
        }
    }
    return true;
}

// funcao auxiliar para carregar o estado do elo, utilizada tanto
// pela CarregarEstado(...) quanto pela CarregarSolucao(...)
bool clEloMaluco::CarregarEstadoAux(ifstream &arquivo)
{
    // uma face auxiliar
    clFace face;
    // contera cada token lido do arquivo
    char str[4];
    // utilizado para ignorar linhas com comentarios
    char lixo[100];

    // laco que le as faces contidas no arquivo
    for(int i = 0; i < numeroDeFaces; i++)
    {
        // le o primeiro token
        arquivo >> str;
        // analisa a primeira letra do token
        switch (str[0])
        {
            // se for um '#' ignora a linha (comentario)
            case '#' :
                arquivo.getline(lixo, sizeof(lixo));
                i--;
                break;
            // se o primeiro caracter do token for 'v'...
            case 'v':
                switch (str[1])
                { // ... verifica o segundo e ajusta a cor da face
                    case 'm' :
                        face.SetCor(VM);
                        break;
                    case 'd' :
                        face.SetCor(VD);
                        break;
                    case 'z' :
                        face.SetCor(VZ);
                        break;
                    default :                           
                        return false;
                }
                break;
            // se for 'a' a cor so pode ser amarela
            case 'a':
                face.SetCor(AM);
                break;
            // se for 'b' a cor so pode ser branca
            case 'b':
                face.SetCor(BR);
                break;
            default :
                return false;
        }
        // ajustando a parte da face
        if (str[0] != '#') 
        // se nao for comentario
        {
            switch (str[2])
            // analisa cada situacao possivel para a parte
            {
                case 's': // parte superior
                    face.SetParte(S);
                    break;
                case 'i': // parte inferior
                    face.SetParte(I);
                    break;
                case 'm': // parte da interseccao entre dois elos
                    face.SetParte(M);
                    break;
                case 'o': // para a face vazia
                    face.SetParte(V);
                    break;
                default :
                    return false;
            }
            // a face eh colocada no elo
            elo[i] = face;
        }
    }
    // carrega a representacao geometrica do elo maluco
    CarregarRepresentacao("elo.dat");
    return true;
}

// Esta funcao ainda nao foi implementada
bool clEloMaluco::GerarSolucao()
{
    /*************************************************************************
    * Quando implementada ela tera que:                                      *
    * . escrever o estado inicial a ser solucionado no comeco de um arquivo  *
    * . escrever os movimentos para se chegar a solucao logo em seguida,     *
    *   usando a seguinte codificacao:                                       *
    *    rsd -> rotacionar parte superior a direita                          *
    *    rse -> rotacionar parte superior a esquerda                         *
    *    rid -> rotacionar parte inferior a direita                          *
    *    rie -> rotacionar parte inferior a esquerda                         *
    *    mfc -> mover face abaixo da face vazia para cima                    *
    *    mfb -> mover face acima da face vazia para baixo                    *
    * . e por fim chamar a funcao abaixo:                                    *
    *    CarregarSolucao( <nome_do_arquivo> );                               *
    *************************************************************************/
    return false;
}
      
// Rotaciona a parte superior para a direita
void clEloMaluco::RodarSupDir()
{
    RodarAux(true, true);
}

// Rotaciona a parte superior para a esquerda
void clEloMaluco::RodarSupEsq()
{
    RodarAux(true, false);
}

// Rotaciona a parte inferior para a direita
void clEloMaluco::RodarInfDir()
{
    RodarAux(false, true);
}

// Rotaciona a parte inferior para a esquerda
void clEloMaluco::RodarInfEsq()
{
    RodarAux(false, false);
}

// funcao auxiliar para rotacionar as partes superior e inferior do elo
// para a esquerda ou para a direita
void clEloMaluco::RodarAux(bool sup, bool dir)
{
    // uma face auxiliar
    clFace face;
    // identificadores de posicao e sinal
    int pos1, pos2, sinal;
    // sinal inicia-se com 1
    sinal = 1;
    
    if(dir) // incializacoes para rotacao a direita
    {
        pos1 = 0;
        pos2 = pos1+3;
        sinal = -sinal;
    }
    else // incializacoes para rotacao a esquerda
    {
        pos2 = 0;
        pos1 = pos2+3;
    }
    
    if(!sup) // incializacoes para rotacao na parte inferior do elo maluco
    {
        pos1+= 12;
        pos2+= 12;
    }
    
    // ajusta cor da face com a da face da posicao2
    face.SetCor(elo[pos2].GetCor());
    face.SetParte(elo[pos2].GetParte());
    
    int i = pos2;
    
    // Troca as faces de posicao de acordo com a direcao da rotacao (direita ou esquerda)
    while(i != pos1)
    {
        elo[i].SetCor(elo[i+sinal].GetCor());
        elo[i].SetParte(elo[i+sinal].GetParte());
        i+= sinal;
    }
    elo[pos1].SetCor(face.GetCor());
    elo[pos1].SetParte(face.GetParte());
}

// Move a face que esta abaixo da face vazia para cima
bool clEloMaluco::MoverFacePCima()
{
    return MoverFaceAux(false);
}

// Move a face que esta acima da face vazia para baixo
bool clEloMaluco::MoverFacePBaixo()
{
    return MoverFaceAux(true);  
}

// Funcao auxiliar para mover a face para cima ou para baixo
bool clEloMaluco::MoverFaceAux(bool baixo)
{
    // recebe a posicao da face vazia no vetor de faces
    int posVazio = EncontrarPosVazio();
    // posicao auxiliar
    int posAux;
    // face auxiliar
    clFace face;    
    
    // inicializando posicao auxiliar
    if(baixo) 
        posAux = posVazio-4;
    else
        posAux = posVazio+4;        
    
    // apenas troca  de lugar a face vazia e a que esta se movimentando
    if ((posVazio > 3 && baixo) || (posVazio < 12 && !baixo))
    {
        face = elo[posVazio];
        elo[posVazio].SetCor(elo[posAux].GetCor());
        elo[posVazio].SetParte(elo[posAux].GetParte());  
        elo[posAux].SetCor(face.GetCor());
        elo[posAux].SetParte(face.GetParte());
        return true;
    }
    return false;
}    

// diz se o elo esta solucionado          
bool clEloMaluco::EstaSolucionado()
{
    // o elo estara solucionado se todas as colunas estiverem solucionadas
    return (ColunaCorreta(0) && ColunaCorreta(1) && ColunaCorreta(2) && ColunaCorreta(3));
}

// retorna se uma coluna do elo esta solucionada
bool clEloMaluco::ColunaCorreta(int coluna)
{
    // cor recebe a cor do primeiro elo da coluna em questao
    ECor cor = elo[coluna].GetCor();
    
    // verifica se as cores estao corretas
    if (cor == BR)
    // se a cor eh branca as 2 faces abaixo tambem tem que ser brancas e a ultima vazia
    {
        if(!(elo[coluna+4].GetCor() == BR && elo[coluna+8].GetCor() == BR && elo[coluna+12].GetCor() == VZ))
            return false;
    }
    else
    {
        if(cor == VZ)
            cor = BR;    
        // verificando se a coluna possui faces da mesma cor
        for(int i = coluna+4; i < coluna+13; i+=4)
        {
            if(cor != elo[i].GetCor())
                return false;
        }
    }
    
    // verifica se as partes do elo estao nas posicoes corretas 
    if(cor != BR)
    {
        // se a cor nao eh branca o elo estara solucionado a face mais alta contem a parte
        // superior as proximas duas a parte de interseccao e a ultima a parte de baixo
        return ( (elo[coluna].GetParte() == S) && (elo[coluna+4].GetParte() == M) &&
                 (elo[coluna+8].GetParte() == M) && (elo[coluna+12].GetParte() == I) );
    }
    else
    {
        // a diferenca eh que o branco possui apenas uma face de interseccao e a coluna
        // possuira a face vazia
        return ( ( (elo[coluna].GetParte() == V) && (elo[coluna+4].GetParte() == S) &&
                   (elo[coluna+8].GetParte() == M) && (elo[coluna+12].GetParte() == I) ) ||
                 ( (elo[coluna].GetParte() == S) && (elo[coluna+4].GetParte() == M) &&
                   (elo[coluna+8].GetParte() == I) && (elo[coluna+12].GetParte() == V)));
    }               
}

// encontra em qual posicao do vetor de faces esta a face vazia
int clEloMaluco::EncontrarPosVazio()
{
    int resposta = 0;
    
    // encontra a posicao da face vazia
    for(int i = 0; i < numeroDeFaces; i++)
    {
        if(elo[i].GetCor() == VZ)
            resposta = i;
    }
    
    if (resposta < 0 || resposta > 15)
        wxMessageBox("H um erro na representao do elo", "Erro de Representao", wxOK | wxICON_ERROR);
    return resposta;
    
}

// a partir de um id devolve uma face por referencia
bool clEloMaluco::CopiarFace(int id, clFace &face)
{
    face.SetCor(elo[id].GetCor());
    face.SetParte(elo[id].GetParte());
    face.SetPonto1(elo[id].GetPonto1());
    face.SetPonto2(elo[id].GetPonto2());
    face.SetPonto3(elo[id].GetPonto3());
    face.SetPonto4(elo[id].GetPonto4());
    return true;
}

// devolve a cabeca da lista de movimentos criada quando
// se gera a solucao automatica
clNohLista<E_MOVIMENTO> *clEloMaluco::CabecaListaSolucao() const
{
    return listaSolucao.Primeiro();
}

// embaralha as faces do elo a partir de um numero "aleatorio"
void clEloMaluco::Embaralhar()
{
    clFace face;
    int j;

    srand(time(0));
    for (int i = 0; i < 16; i++)
    {
        j = rand() % 16;
        face.Copiar(elo[i]);
        elo[i].Copiar(elo[j]);
        elo[j].Copiar(face);
    }
}

#endif

      
