/********************************************************************
*                         ELO MALUCO                                *
* ----------------------------------------------------------------- *
*      Desenvolvedores:                                             *
*               Anderson de Rezende Rocha - DCC/UFLA                *
*               undersun@comp.ufla.br                               *
*                                                                   *
*               Jlio Csar Alves - DCC/UFLA                        *
*               jcalves@comp.ufla.br                                *
* ----------------------------------------------------------------- *
*    COMPUTACAO GRAFICA: Prof. Bruno Oliveira Schneider             *
********************************************************************/

/********************************************************************
*  Este arquivo contem a classe EloMalucoGLcanvas que define a      *
*  parte de OpenGL do Elo Maluco, ou seja o canvas que desenha o    *
*  Elo Maluco na tela                                               *
********************************************************************/

#ifndef ELO_GL_CANVAS_H
#define ELO_GL_CANVAS_H

// fstream -> utilizada para trabalhar com arquivos
#include <fstream>
// elomaluco.h -> define o Elo Maluco
#include "elomaluco.h"
// glcanvas.h -> herdada pela classe EloMalucoGLCanvas
#include <wx/glcanvas.h>

// Um pixel sera representado por 3 valores (RGB)
typedef GLubyte Pixel[3];

// definicao da classe EloMalucoGLCanvas
class EloMalucoGLCanvas : public wxGLCanvas
{
    public:
        // Construtor
        EloMalucoGLCanvas(wxWindow *parent, const wxWindowID id = -1,
            const wxPoint& pos = wxDefaultPosition, const wxSize& size = wxDefaultSize,
            long style = 0, const wxString& name = "EloMalucoGLCanvas");
        // Destrutor
        ~EloMalucoGLCanvas();

        // chamada pelo sistema sempre que o Elo Maluco precisa ser redesenhado
        void OnPaint(wxPaintEvent& event);
        
        // funcao chamada toda vez o Elo Maluco precisa ser redesenhado
        void Desenhar();
        
        // chamada quando a janela tem seu tamanho alterado
        void OnSize(wxSizeEvent& event);
        // chamada para evitar que a janela fique piscando
        void OnEraseBackground(wxEraseEvent& event);
        // chamada para que a janela possa receber dados do teclado
        void AoEntrarNaJanela( wxMouseEvent& event );
        // chamada quando alguma tecla eh apertada
        void OnTeclaPressionada(wxKeyEvent& event);

        // faz o Elo Maluco parar ou ficar girando
        void PararGirar();
        // faz o Elo Maluco girar uma determinada quantidade para a esquerda
        void GirarEsq();
        // faz o Elo Maluco girar uma determinada quantidade para a direita
        void GirarDir();

        // embaralha o Elo Maluco
        void EmbaralharElo();
        // salva o estado atual do Elo Maluco em um arquivo
        void SalvarEstado(wxString arquivo);
        // carrega o estado atual do Elo Maluco a partir de um arquivo
        void CarregarEstado(wxString arquivo);

        // rotaciona a parte superior do Elo Maluco a direita
        void RodarSupDir();
        // rotaciona a parte superior do Elo Maluco a esquerda
        void RodarSupEsq();
        // rotaciona a parte inferior do Elo Maluco a direita
        void RodarInfDir();
        // rotaciona a parte inferior do Elo Maluco a esquerda
        void RodarInfEsq();
        // move a face abaixo da face vazia para cima
        void MoverFacePCima();
        // move a face acima da face vazia para baixo
        void MoverFacePBaixo();

        // faz um movimento contido na lista de movimentos de solucao
        bool AvancarSolucao();
        // gera uma solucao para o estado atual do Elo Maluco
        bool GerarSolucao();

        // inicia o OpenGL, texturas, etc.
        bool IniciarOpenGL();
        // variavel que guarda que o Elo Maluco esta em modo solucao automatica, ou seja,
        // nao esta jogavel
        bool modoSolucaoAutomatica;

    private:

        // diz se o OpenGL foi iniciado
        bool openGLIniciado;
        // diz se as texturas foram corretamente carregadas
        bool texturasOk;
        // retorna a maior potencia de 2 menor ou igual a x
        int TamanhoPermitido(int x);
        // carrega uma imagem a partir de um arquivo no formato PPM
        bool CarregarImagem(char *nomeDoArquivo);
        // guardam a altura e a largura de uma imagem
        int largura, altura;
        // guarda o quanto o Elo Maluco vai girar
        int graus;
        // diz se o Elo Maluco esta girando ou parada
        bool girando;

        // o Elo Maluco propriamente dito
        clEloMaluco elo;

        // noh que guardara o primeiro movimento da lista de solucao automatica
        clNohLista<E_MOVIMENTO> *nohListaSolucao;

        // guardara uma imagem formada de Pixels (3 valores GLubyte para cada pixel)
        Pixel *imagem;

        // vetor que guarda os identificadores das texturas utilizadas
        // 3 texturas para cada cor + face vazia + topo = 14 texturas
        GLuint idsTexturas[14];

        DECLARE_EVENT_TABLE()
};

//A tabela de eventos conecta os eventos wxWindows com as funes handlers que processaro estes eventos.

BEGIN_EVENT_TABLE(EloMalucoGLCanvas, wxGLCanvas)
    EVT_SIZE(EloMalucoGLCanvas::OnSize)
    EVT_PAINT(EloMalucoGLCanvas::OnPaint)
    EVT_ERASE_BACKGROUND(EloMalucoGLCanvas::OnEraseBackground)
    EVT_ENTER_WINDOW( EloMalucoGLCanvas::AoEntrarNaJanela )
    EVT_KEY_DOWN(EloMalucoGLCanvas::OnTeclaPressionada)
END_EVENT_TABLE()

// construtor
EloMalucoGLCanvas::EloMalucoGLCanvas(wxWindow *parent, wxWindowID id,
    const wxPoint& pos, const wxSize& size, long style, const wxString& name):
  wxGLCanvas(parent, (wxGLCanvas*) NULL, id, pos, size, style, name )
{
    // inicializacoes
    openGLIniciado = false;
    imagem = NULL;
    graus = 0;
    girando = false;
    modoSolucaoAutomatica = false;
    nohListaSolucao = NULL;
    // carrega o estado do elo maluco a partir do arquivo 'estado.elo'
    elo.CarregarEstado("estado.elo");
    // embaralha as faces do elo
    elo.Embaralhar();
}

// destrutor5
EloMalucoGLCanvas::~EloMalucoGLCanvas()
{
}

// possibilita a entrada de dados pelo teclado
void EloMalucoGLCanvas::AoEntrarNaJanela( wxMouseEvent& event )
{
    SetFocus();
}

// A funo no faz nada mas deve ser declarada para evitar que a janela fique piscando
void EloMalucoGLCanvas::OnEraseBackground(wxEraseEvent& event)
{
}

// chamada quando alguma tecla eeh pressionada
void EloMalucoGLCanvas::OnTeclaPressionada(wxKeyEvent& event)
{
    // TECLAS:
    // Q -> rotaciona parte superior a esquerda
    // E -> rotaciona parte superior a direita
    // A -> rotaciona parte inferior a esquerda
    // D -> rotaciona parte inferior a direita
    // W -> move face abaixo da face vazia para cima
    // W -> move face acima da fave vazia para baixo

    // SETA para a ESQUERDA -> gira o Elo Maluco um pouco para a esquerda
    // SETA para a DIREITA  -> gira o Elo Maluco um pouco para a direita
    switch(event.GetKeyCode())
    {
        case 'Q' :
            if (!modoSolucaoAutomatica)
                RodarSupEsq();
            break;
        case 'W' :
            if (!modoSolucaoAutomatica)
                MoverFacePCima();
            break;
        case 'E' :
            if (!modoSolucaoAutomatica)
                RodarSupDir();
            break;
        case 'A' :
            if (!modoSolucaoAutomatica)
                RodarInfEsq();
            break;
        case 'S' :
            if (!modoSolucaoAutomatica)
                MoverFacePBaixo();
            break;
        case 'D' :
            if (!modoSolucaoAutomatica)
                RodarInfDir();
            break;
        case WXK_LEFT :
            GirarEsq();
            break;
        case WXK_RIGHT :
            GirarDir();
            break;
    }
}

// embaralha as faces do Elo Maluco
void EloMalucoGLCanvas::EmbaralharElo()
{
    elo.Embaralhar();
    Desenhar();
}

// salva o estado do Elo Maluco em um arquivo
void EloMalucoGLCanvas::SalvarEstado(wxString arquivo)
{
    elo.SalvarEstado((char *)arquivo.c_str());
}

// carrega o estado do Elo Maluco a partir de um arquivo
void EloMalucoGLCanvas::CarregarEstado(wxString arquivo)
{
    elo.CarregarEstado((char *)arquivo.c_str());
    Desenhar();
}

// faz o Elo Maluco ficar girando ou parar de girar
void EloMalucoGLCanvas::PararGirar()
{
    if(girando)
        graus = 0;
    else
        graus = -5;

    girando = !girando;
    Desenhar();
}

// gira o Elo Maluco um pouco para a esquerda
void EloMalucoGLCanvas::GirarEsq()
{
    graus = 10;
    Desenhar();
    if (!girando)
        graus = 0;
}

// gira o Elo Maluco um pouco para a direita
void EloMalucoGLCanvas::GirarDir()
{
    graus = -10;
    Desenhar();
    if (!girando)
        graus = 0;
}

// rotaciona a parte superior do Elo Maluco para a direita
void EloMalucoGLCanvas::RodarSupDir()
{
    elo.RodarSupDir();
    Desenhar();
}

// rotaciona a parte superior do Elo Maluco para a esquerda
void EloMalucoGLCanvas::RodarSupEsq()
{
    elo.RodarSupEsq();
    Desenhar();
}

// rotaciona a parte inferior do Elo Maluco para a direita
void EloMalucoGLCanvas::RodarInfDir()
{
    elo.RodarInfDir();
    Desenhar();
}

// rotaciona a parte inferior do Elo Maluco para a esquerda
void EloMalucoGLCanvas::RodarInfEsq()
{
    elo.RodarInfEsq();
    Desenhar();
}

// move a parte abaixo da face vazia para cima
void EloMalucoGLCanvas::MoverFacePCima()
{
    elo.MoverFacePCima();
    Desenhar();
}

// move a parte acima da face vazia para baixo
void EloMalucoGLCanvas::MoverFacePBaixo()
{
    elo.MoverFacePBaixo();
    Desenhar();
}

// funcao chamada pelo sistema toda vez o Elo Maluco precisa ser redesenhado
void EloMalucoGLCanvas::OnPaint( wxPaintEvent& event )
{
    // face auxiliar
    clFace face;

    // criado para que se possa desenhar
     wxPaintDC dc(this);
    //wxClientDC dc(this);

    // Configura este canvas como local onde sero chamadas as funes OpenGL
    SetCurrent();

    // inicializa o OpenGL se ainda nao o foi
    if (!openGLIniciado)
    {
        openGLIniciado = true;
        if (!IniciarOpenGL())
            texturasOk = false;
        else
            texturasOk = true;
    }

    // Limpa a tela
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // rotaciona o Elo Maluco tantos graus em torno do eixo Y
    glRotatef(graus, 0.0, 1.0, 0.0);

    if(texturasOk)
    // se as texturas foram corretamente carregadas
    {
        // Desenha o topo do Elo Maluco

        // a textura do topo eh a de numero 13
        glBindTexture ( GL_TEXTURE_2D, idsTexturas[13]);
        // Desenha a face do topo do Elo Maluco
        // delimitada pelos pontos abaixo
        glBegin(GL_QUADS);
            glTexCoord2f(0.0f, 0.0f);
            glVertex3d(-0.3, 0.9, -0.3);

            glTexCoord2f(0.0f, 1.0f);
            glVertex3d(0.3, 0.9, -0.3);

            glTexCoord2f(1.0f, 1.0f);
            glVertex3d(0.3, 0.9, 0.3);

            glTexCoord2f(1.0f, 0.0f);
            glVertex3d(-0.3, 0.9, 0.3);

        glEnd();

        // laco que desenha as faces que realmente pertencem ao Elo Maluco
        // sabemos que sao 16 faces
        for (int i = 0; i < 16; i++)
        {
            // a face auxiliar recebe os valores da face real a ser desenhada
            elo.CopiarFace(i, face);
            // a textura correta eh escolhida
            glBindTexture ( GL_TEXTURE_2D, idsTexturas[face.GetIdTextura()]);
            // desenha a face de acordo com os seus pontos
            glBegin(GL_QUADS);
                glTexCoord2f(0.0f, 0.0f);
                glVertex3d(face.GetPonto1().GetX(), face.GetPonto1().GetY(),
                            face.GetPonto1().GetZ());

                glTexCoord2f(0.0f, 1.0f);
                glVertex3d(face.GetPonto2().GetX(), face.GetPonto2().GetY(),
                            face.GetPonto2().GetZ());

                glTexCoord2f(1.0f, 1.0f);
                glVertex3d(face.GetPonto3().GetX(), face.GetPonto3().GetY(),
                                face.GetPonto3().GetZ());

                glTexCoord2f(1.0f, 0.0f);
                glVertex3d(face.GetPonto4().GetX(), face.GetPonto4().GetY(),
                            face.GetPonto4().GetZ());
            glEnd();
        }
    }

    if(elo.EstaSolucionado() && !modoSolucaoAutomatica && graus != 40)
    {
        graus = 40;

        // guarda a matriz de transformacao
        glPushMatrix();

        // gera um pequeno efeito especial
        for(int i = 0; i < 10; i++)
        {
            glRotatef(30,1,0,0);
            glRotatef(20,0,0,1);
            if (i % 2 == 0)
                glClearColor(0,0,0,0);
            else
                glClearColor(1,1,1,0);
            OnPaint(event);
        }

        // recupera a matriz de transfomacao anterior
        glPopMatrix();
        glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
        OnPaint(event);
        if (!girando)
            graus = 0;
    }
    // Forca o sistema a execultar os comandos definidos previamente o mais rapido possivel
    glFlush();
    // Mostra os comandos OpenGL anteriores na janela
    SwapBuffers();
}

// A funcao abaixo eh igual a anterior, com a excecao de criar um wxClientDC ao inves de 
// um wxPaintDC. Porque isso?
// Para manter a compatibilidade. Em Linux a funcao OnPaint poderia ser chamada diretamente
// por qualquer outra que funcionaria normalmente. Mas em WindowsNT e 2000 a tela nao eh
// redesenhada. Procuramos nos informar e descobrimos que o correto seria chamar a 
// funcao Refresh(); mas ao fazer isso vimos que ficou perfeito em Windows, mas no Linux a 
// cada vez que a tela precisava ser redesenhada o Elo piscava, ocasionando um efeito muito
// ruim.
// Por fim, esta redundancia de funcao pode nao ser a melhor alternativa, mas permite que
// o programa seja compilado, funcionando corretamente, nos dois sistemas operacionais sem 
// que seja preciso modificar um caracter de codigo.

// funcao chamada toda vez o Elo Maluco precisa ser redesenhado
void EloMalucoGLCanvas::Desenhar()
{
    // face auxiliar
    clFace face;

    // criado para que se possa desenhar
    wxClientDC dc(this);

    // Configura este canvas como local onde sero chamadas as funes OpenGL
    SetCurrent();

    // inicializa o OpenGL se ainda nao o foi
    if (!openGLIniciado)
    {
        openGLIniciado = true;
        if (!IniciarOpenGL())
            texturasOk = false;
        else
            texturasOk = true;
    }

    // Limpa a tela
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // rotaciona o Elo Maluco tantos graus em torno do eixo Y
    glRotatef(graus, 0.0, 1.0, 0.0);

    if(texturasOk)
    // se as texturas foram corretamente carregadas
    {
        // Desenha o topo do Elo Maluco

        // a textura do topo eh a de numero 13
        glBindTexture ( GL_TEXTURE_2D, idsTexturas[13]);
        // Desenha a face do topo do Elo Maluco
        // delimitada pelos pontos abaixo
        glBegin(GL_QUADS);
            glTexCoord2f(0.0f, 0.0f);
            glVertex3d(-0.3, 0.9, -0.3);

            glTexCoord2f(0.0f, 1.0f);
            glVertex3d(0.3, 0.9, -0.3);

            glTexCoord2f(1.0f, 1.0f);
            glVertex3d(0.3, 0.9, 0.3);

            glTexCoord2f(1.0f, 0.0f);
            glVertex3d(-0.3, 0.9, 0.3);

        glEnd();

        // laco que desenha as faces que realmente pertencem ao Elo Maluco
        // sabemos que sao 16 faces
        for (int i = 0; i < 16; i++)
        {
            // a face auxiliar recebe os valores da face real a ser desenhada
            elo.CopiarFace(i, face);
            // a textura correta eh escolhida
            glBindTexture ( GL_TEXTURE_2D, idsTexturas[face.GetIdTextura()]);
            // desenha a face de acordo com os seus pontos
            glBegin(GL_QUADS);
                glTexCoord2f(0.0f, 0.0f);
                glVertex3d(face.GetPonto1().GetX(), face.GetPonto1().GetY(),
                            face.GetPonto1().GetZ());

                glTexCoord2f(0.0f, 1.0f);
                glVertex3d(face.GetPonto2().GetX(), face.GetPonto2().GetY(),
                            face.GetPonto2().GetZ());

                glTexCoord2f(1.0f, 1.0f);
                glVertex3d(face.GetPonto3().GetX(), face.GetPonto3().GetY(),
                                face.GetPonto3().GetZ());

                glTexCoord2f(1.0f, 0.0f);
                glVertex3d(face.GetPonto4().GetX(), face.GetPonto4().GetY(),
                            face.GetPonto4().GetZ());
            glEnd();
        }
    }

    if(elo.EstaSolucionado() && !modoSolucaoAutomatica && graus != 40)
    {
        graus = 40;

        // guarda a matriz de transformacao
        glPushMatrix();

        // gera um pequeno efeito especial
        for(int i = 0; i < 10; i++)
        {
            glRotatef(30,1,0,0);
            glRotatef(20,0,0,1);
            if (i % 2 == 0)
                glClearColor(0,0,0,0);
            else
                glClearColor(1,1,1,0);
            Desenhar();
        }

        // recupera a matriz de transfomacao original
        glPopMatrix();
        glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
        Desenhar();
        if (!girando)
            graus = 0;
    }
    // Forca o sistema a execultar os comandos definidos previamente o mais rapido possivel
    glFlush();
    // Mostra os comandos OpenGL anteriores na janela
    SwapBuffers();
}

// gera uma solucao para o estado atual do elo
bool EloMalucoGLCanvas::GerarSolucao()
{
    wxPaintEvent event;
    if(elo.GerarSolucao())
    {
        modoSolucaoAutomatica = true;
        nohListaSolucao = elo.CabecaListaSolucao();

        OnPaint(event);
        return true;
    }
    return false;
}

// avanca um movimento para solucionar o Elo Maluco
bool EloMalucoGLCanvas::AvancarSolucao()
{
    if(nohListaSolucao != NULL)
    // se ha um movimento a ser feito...
    {
        // ... o executa
        switch(nohListaSolucao->Valor())
        {
            case RSE:
                RodarSupEsq();
                break;
            case RSD:
                RodarSupDir();
                break;
            case RIE:
                RodarInfEsq();
                break;
            case RID:
                RodarInfDir();
                break;
            case MFC:
                MoverFacePCima();
                break;
            case MFB:
                MoverFacePBaixo();
                break;
            default:
                break;
        }

        nohListaSolucao = nohListaSolucao->Proximo();

        // acabaram-se os movimentos, eh verificado se o elo realmente foi solucionado
        if(nohListaSolucao == NULL)
        {
            if (elo.EstaSolucionado())
            {
                wxMessageBox("Elo Solucionado!", "Elo Maluco", wxOK);
                return false;
            }
            else
            {
                wxMessageBox("Arquivo de soluo no  vlido.\n"
                             "Impossvel mostrar os passos da soluo.",
                             "Elo Maluco", wxOK | wxICON_INFORMATION);
            }
            return false;
        }
    }
    return true;
}

// retorna a maior potencia de 2 menor ou igual a x
int EloMalucoGLCanvas::TamanhoPermitido(int x)
{
    int r;

    r = 1;
    while(r < x) r=(r<<1);

    if(r==x) return r;
    else return r>>1;
}

// carrega uma imagem a partir de um arquivo no formato PPM
bool EloMalucoGLCanvas::CarregarImagem(char *nomeDoArquivo)
{
    // largura, altura e intensidade maxima da figura
    int larg,alt,max;
    // os valores RGB de um pixel
    unsigned int r,g,b;
    // auxiliar
    int k;
    // utilizado para ignorar uma linha
    char linha[100];
    // string mostrada quando ha algum erro
    wxString mensagemDeErro;

    ifstream arquivo(nomeDoArquivo);

    if (!arquivo)
    {
        mensagemDeErro = "Arquivo ";
        mensagemDeErro << nomeDoArquivo;
        mensagemDeErro << " no pde ser aberto";
        wxMessageBox(mensagemDeErro, "Elo Maluco", wxOK | wxICON_ERROR);
        return false;
    }
    else // se o arquivo pode ser aberto
    {
        // le-se a primeira linha
        arquivo.getline(linha, sizeof(linha));

        // verifica-se o tipo de arquivo
        if (linha[0] != 'P' || linha[1] != '3')
        {
            mensagemDeErro = "Arquivo ";
            mensagemDeErro << nomeDoArquivo;
            mensagemDeErro << " no possui formato vlido";
            wxMessageBox(mensagemDeErro, "Elo Maluco", wxOK | wxICON_ERROR);
            return false;
        }
        else
        {
            // ignorando comentarios
            arquivo.getline(linha, sizeof(linha));

            // le a largura da imagem
            arquivo >> larg;
            // le a altura da imagem
            arquivo >> alt;
            // le a intensidade maxima, nao sera usada
            arquivo >> max;

            // como para o OpenGL a altura e a largura tem que ser potencia de dois, pega-se maior parte
            // da imagem com essas dimensoes
            largura = TamanhoPermitido(larg);
            altura = TamanhoPermitido(alt);

            // aloca espaco para a imagem
            imagem = new Pixel[altura*largura*sizeof(Pixel)];

            // laco que carrega a imagem
            for(int i = 0; i < altura; ++i)
            {
                for(int j = 0; j < largura; ++j)
                {
                    // le os valores RGB de cada pixel
                    arquivo >> r >> g >> b;
                    k = i*largura + j;
                    (*(imagem+k))[0] = (GLubyte)r;
                    (*(imagem+k))[1] = (GLubyte)g;
                    (*(imagem+k))[2] = (GLubyte)b;
                }
                // ignora o que sobrou na largura da imagem
                for(int j = largura; j < larg; ++j)
                    arquivo >> r >> g >> b;
            }
        }
    }
    return true;
}

// Inicializacoes do OpenGL
bool EloMalucoGLCanvas::IniciarOpenGL()
{
    // Configura este canvas como local onde sero chamadas as funes OpenGL
    SetCurrent();

    // Ajusta a cor de fundo deste canvas
    // neste caso cinza escuro
    glClearColor(0.4f, 0.4f, 0.4f, 0.0f); 

    // tratamento de faces escondidas
    glClearDepth(1.0f);
    glEnable( GL_DEPTH_TEST);

    // guardara o nome de cada arquivo
    char *arquivo = "";

    glPixelStorei(GL_UNPACK_ALIGNMENT,1);

    // habilita o uso de texturas
    glGenTextures (1, idsTexturas);
    for(int i = 0; i < 14; i++)
        idsTexturas[i] = i;

    // laco que chama o carregamento das figuras
    for(int i = 0; i < 14; i++)
    {
        switch(i)
        {
            case AMS:
                arquivo = "ams.ppm";
                break;
            case AMM:
                arquivo = "amm.ppm";
                break;
            case AMI:
                arquivo = "ami.ppm";
                break;
            case BRS:
                arquivo = "brs.ppm";
                break;
            case BRM:
                arquivo = "brm.ppm";
                break;
            case BRI:
                arquivo = "bri.ppm";
                break;
            case VDS:
                arquivo = "vds.ppm";
                break;
            case VDM:
                arquivo = "vdm.ppm";
                break;
            case VDI:
                arquivo = "vdi.ppm";
                break;
            case VMS:
                arquivo = "vms.ppm";
                break;
            case VMM:
                arquivo = "vmm.ppm";
                break;
            case VMI:
                arquivo = "vmi.ppm";
                break;
            case VZO:
                arquivo = "vzo.ppm";
                break;
            case 13:
                arquivo = "topo.ppm";
                break;
        }

        // se ha Lixo na imagem
        if (imagem != NULL)
            delete imagem;

        if (!CarregarImagem(arquivo))
        {
            wxString mensagemDeErro;
            mensagemDeErro = "Impossvel carregar texturas do Elo Maluco.";
            wxMessageBox(mensagemDeErro, "Elo Maluco", wxOK | wxICON_ERROR);
            return false;
        }
        else // se a imagem foi corretamente carregada
        {
            // diz qual id dos idsTexturas sera desta imagem
            glBindTexture ( GL_TEXTURE_2D, idsTexturas[i]);
            // garda a textura no buffer de texturas
            glTexImage2D(GL_TEXTURE_2D, 0, 3, (GLsizei)largura, (GLsizei)altura,
                         0, GL_RGB, GL_UNSIGNED_BYTE,imagem);
            // assegura que a imagem nao eh wraped
            glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
            glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);

            // escolhe o tipo de mapeamento
            glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
            glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

            // GL_DECAL - manda sobrescrever os pixels com a textura
            glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_DECAL);
        }
    }

    // descaloca a imagem
    if (imagem != NULL)
        delete imagem;

    // habilita a texturizacao
    glEnable(GL_TEXTURE_2D);
    // rotacao em X para dar um melhor angulo
    glRotatef(-20.0, 1.0, 0.0, 0.0);
    // rotacao inicial em Y
    glRotatef(30, 0.0, 1.0, 0.0);
    return true;
}


// Quando a janela  redimensionada, nao faz nada. Mas se ela nao existir
// o elo desaperece
void EloMalucoGLCanvas::OnSize(wxSizeEvent& event)
{
}

#endif

