#include "image.h"

Image *CreateImage(int ncols, int nrows)
{
  Image *img=NULL;
  int i;

  img = (Image *) calloc(1,sizeof(Image));
  if (img == NULL){
    Error(MSG1,"CreateImage");
  }

  img->val   = AllocIntArray(nrows*ncols);
  img->tbrow = AllocIntArray(nrows);

  img->tbrow[0]=0;
  for (i=1; i < nrows; i++)
    img->tbrow[i]=img->tbrow[i-1]+ncols;
  img->ncols = ncols;
  img->nrows = nrows;
 
 return(img);
}

void DestroyImage(Image **img)
{
  Image *aux;

  aux = *img;
  if(aux != NULL){
    if (aux->val != NULL)   free(aux->val); 
    if (aux->tbrow != NULL) free(aux->tbrow);
    free(aux);    
    *img = NULL;
  }
}

Image *ReadImage(char *filename)
{
  FILE *fp=NULL;
  unsigned char *value=NULL;
  char type[10];
  int  i,ncols,nrows,n;
  Image *img=NULL;
  char z[256];

  fp = fopen(filename,"rb");
  if (fp == NULL){
    fprintf(stderr,"Cannot open %s\n",filename);
    exit(-1);
  }
  fscanf(fp,"%s\n",type);
  if((strcmp(type,"P5")==0)){
    NCFgets(z,255,fp);
    sscanf(z,"%d %d\n",&ncols,&nrows);
    n = ncols*nrows;
    NCFgets(z,255,fp);
    sscanf(z,"%d\n",&i);
    value = (unsigned char *)calloc(n,sizeof(unsigned char));
    if (value != NULL){
      fread(value,sizeof(unsigned char),n,fp);
    }else{
      fprintf(stderr,"Insufficient memory in ReadImage\n");
      exit(-1);
    }
    fclose(fp);
    img = CreateImage(ncols,nrows);
    for (i=0; i < n; i++)
      img->val[i]=(int)value[i];
    free(value);
  }else{
    if((strcmp(type,"P2")==0)){
      NCFgets(z,255,fp);
      sscanf(z,"%d %d\n",&ncols,&nrows);
      n = ncols*nrows;
      NCFgets(z,255,fp);
      sscanf(z,"%d\n",&i);
      img = CreateImage(ncols,nrows);
      for (i=0; i < n; i++)
	fscanf(fp,"%d",&img->val[i]);
      fclose(fp);
    }else{
      fprintf(stderr,"Input image must be P2 or P5\n");
      exit(-1);
    }
  }

  return(img);
}

void WriteImage(Image *img,char *filename)
{
  FILE *fp;
  int i, n, Imax;

  fp = fopen(filename,"wb");
  if (fp == NULL){
    fprintf(stderr,"Cannot open %s\n",filename);
    exit(-1);
  }
  n    = img->ncols*img->nrows;
  if ((Imax=MaximumValue(img))==INT_MAX){
    Warning("Image with infinity values","WriteImage");
    Imax = INT_MIN;
    for (i=0; i < n; i++) 
      if ((img->val[i] > Imax)&&(img->val[i]!=INT_MAX))
	Imax = img->val[i];
    fprintf(fp,"P2\n");
    fprintf(fp,"%d %d\n",img->ncols,img->nrows);
    fprintf(fp,"%d\n",Imax+1);
  } else {
    fprintf(fp,"P2\n");
    fprintf(fp,"%d %d\n",img->ncols,img->nrows);
    if (Imax==0) Imax++;
    fprintf(fp,"%d\n",Imax);
  }
 
  for (i=0; i < n; i++) {
    if (img->val[i]==INT_MAX)
      fprintf(fp,"%d ",Imax+1);
    else
      fprintf(fp,"%d ",img->val[i]);
    if (((i+1)%17) == 0)
      fprintf(fp,"\n");
  }

  fclose(fp);

}

int MinimumValue(Image *img)
{
  int i,min,n;

  n = img->ncols*img->nrows;
  min = img->val[0];
  for (i=1; i < n; i++)
    if (img->val[i] < min)
      min = img->val[i];

  return(min);
}

int MaximumValue(Image *img)
{
  unsigned int i, n, r;
  int max;

  max = img->val[0];
  n = img->ncols*img->nrows - 1;
  r = n%4;
  n -= r;
  for (i=1; i < n; i+=4) {
    if (img->val[i] > max)
      max = img->val[i];
    if (img->val[i+1] > max)
      max = img->val[i+1];
    if (img->val[i+2] > max)
      max = img->val[i+2];
    if (img->val[i+3] > max)
      max = img->val[i+3];
  }
  while (r != 0) {
    if (img->val[i+r-1] > max)
      max = img->val[i+r-1];
    --r;
  }

  return(max);
}

char ValidPixel(Image *img, int x, int y)
{
  if ((x >= 0)&&(x < img->ncols)&&
      (y >= 0)&&(y < img->nrows))
    return(1);
  else
    return(0);
}



Image *Rotate(Image *img, float theta) 
{
  float R[2][2],x,y,d1,d2,d3,d4,Ix1,Ix2,If;
  Image *rot;
  Pixel u,v,prev,next;
  int diag;

  theta = theta*PI/180.0;
  R[0][0] = cos(theta);
  R[0][1] = sin(theta);
  R[1][0] = -sin(theta);
  R[1][1] = cos(theta);

  diag = sqrt(img->ncols*img->ncols + img->nrows*img->nrows);

  rot = CreateImage(diag,diag); 

  for (v.y=0; v.y < rot->nrows; v.y++)
    for (v.x=0; v.x < rot->ncols; v.x++){
      x = ((v.x-rot->ncols/2.)*R[0][0] + (v.y-rot->nrows/2.)*R[0][1]) 
	+ img->ncols/2.;
      y = ((v.x-rot->ncols/2.)*R[1][0] + (v.y-rot->nrows/2.)*R[1][1]) 
	+ img->nrows/2.;
      u.x = (int)(x+0.5);
      u.y = (int)(y+0.5);
      if (ValidPixel(img,u.x,u.y)){
	if (x < u.x) {
	  next.x = u.x;
	  prev.x = u.x - 1;
	} else {
	  next.x = u.x + 1;
	  prev.x = u.x;
	}
	d1 = next.x - x;
	d2 = x - prev.x;
	if (y < u.y) {
	  next.y = u.y;
	  prev.y = u.y - 1;
	} else {
	  next.y = u.y + 1;
	  prev.y = u.y;
	}
	d3 = next.y - y;
	d4 = y - prev.y;

	if (ValidPixel(img,prev.x,prev.y)&&ValidPixel(img,next.x,prev.y))
	  Ix1 = d1*img->val[prev.x+img->tbrow[prev.y]] + 
	    d2*img->val[next.x+img->tbrow[prev.y]];
	else
	  Ix1 = img->val[u.x+img->tbrow[u.y]];

	if (ValidPixel(img,prev.x,next.y)&&ValidPixel(img,next.x,next.y))
	  Ix2 = d1*img->val[prev.x+img->tbrow[next.y]] + 
	    d2*img->val[next.x+img->tbrow[next.y]];
	else
	  Ix2 = img->val[u.x+img->tbrow[u.y]];
	
	If = d3*Ix1 + d4*Ix2;

	rot->val[v.x+rot->tbrow[v.y]] = (int)If;
      }
    }
  
  return(rot);
}

Image *Scale(Image *img, float Sx, float Sy) 
{
  float S[2][2],x,y,d1,d2,d3,d4,Ix1,Ix2,If;
  Image *scl;
  Pixel u,v,prev,next;
  
  if (Sx == 0.0) Sx = 1.0;
  if (Sy == 0.0) Sy = 1.0;

  S[0][0] = 1.0/Sx;
  S[0][1] = 0;
  S[1][0] = 0;
  S[1][1] = 1.0/Sy;

  scl = CreateImage((int)(img->ncols*fabs(Sx) + 0.5),(int)(img->nrows*fabs(Sy) + 0.5)); 

  for (v.y=0; v.y < scl->nrows; v.y++)
    for (v.x=0; v.x < scl->ncols; v.x++){
      x = ((v.x-scl->ncols/2.)*S[0][0] + (v.y-scl->nrows/2.)*S[0][1]) 
	+ img->ncols/2.;
      y = ((v.x-scl->ncols/2.)*S[1][0] + (v.y-scl->nrows/2.)*S[1][1]) 
	+ img->nrows/2.;
      u.x = (int)(x+0.5);
      u.y = (int)(y+0.5);
      if (ValidPixel(img,u.x,u.y)){
	if (x < u.x) {
	  next.x = u.x;
	  prev.x = u.x - 1;
	} else {
	  next.x = u.x + 1;
	  prev.x = u.x;
	}
	d1 = next.x - x;
	d2 = x - prev.x;
	if (y < u.y) {
	  next.y = u.y;
	  prev.y = u.y - 1;
	} else {
	  next.y = u.y + 1;
	  prev.y = u.y;
	}
	d3 = next.y - y;
	d4 = y - prev.y;

	if (ValidPixel(img,prev.x,prev.y)&&ValidPixel(img,next.x,prev.y))
	  Ix1 = d1*img->val[prev.x+img->tbrow[prev.y]] + 
	    d2*img->val[next.x+img->tbrow[prev.y]];
	else
	  Ix1 = img->val[u.x+img->tbrow[u.y]];

	if (ValidPixel(img,prev.x,next.y)&&ValidPixel(img,next.x,next.y))
	  Ix2 = d1*img->val[prev.x+img->tbrow[next.y]] + 
	    d2*img->val[next.x+img->tbrow[next.y]];
	else
	  Ix2 = img->val[u.x+img->tbrow[u.y]];
	
	If = d3*Ix1 + d4*Ix2;

	scl->val[v.x+scl->tbrow[v.y]] = (int)If;
      }
    }
  
  return(scl);
}

Image *CopyImage(Image *img)
{
  Image *imgc;

  imgc = CreateImage(img->ncols,img->nrows);
  memcpy(imgc->val,img->val,img->ncols*img->nrows*sizeof(int));
  
  return(imgc);
}
